// -----------------------------------------------------------------------------
// Project:	TToolbarEditor
// Module:	tbedit - Main component and run time functionality
// Description:	Toolbar editor/designer
// Version:	01.00
// Date:	15-AUG-1996
// Target:	Windows 95, Delphi 2.0
// Author:	Anders Melander, anme@biocat.ruc.dk
// Copyright	(c) 1996 by Anders Melander
// Formatting:	2 space indent, 8 space tabs, 80 columns.
// -----------------------------------------------------------------------------
// See tbEdit.hlp for further information.
// -----------------------------------------------------------------------------
// Revision history:
// 0100	150896	anme	Initial release.
// -----------------------------------------------------------------------------
unit tbedit;

// -----------------------------------------------------------------------------
//
//			Interface
//
// -----------------------------------------------------------------------------
interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Menus, extctrls, buttons;

type
  TToolbarEditor = class;
  TObjectList = class;

  TItemMoveState = (msMoving, msAdding);

  TToolbarItem = class(TComponent)
    private
      FGlyph			: TBitmap;
      FNumGlyphs		: Integer;
      FEnabled			: Boolean;
      FVisible			: Boolean;
      FDescription		: String;
      FHint			: String;
      FTop			,
      FLeft			,
      FWidth			,
      FHeight			: Integer;
      FMenuItem			: TMenuItem;
      FToolButton		: TComponent;
      FCategoryIndex		: Integer;
      FToolbarEditor		: TToolbarEditor;
      FMoveState		: TItemMoveState;
      // FMultiInstance		: Boolean;

      function StoreButton: boolean;

    protected
      procedure SetGlyph(value: TBitmap);
      procedure SetToolButton(value: TComponent);
      procedure SetCategoryIdx(idx: Integer);
      procedure SetCategory(value: String);
      function GetCategory: String;
      procedure SetMenuItem(Value: TMenuItem);

      procedure DrawDragImage(Pos: TPoint);
      procedure RemoveDragImage;
      procedure FrameRect(Handle: HDC; R: TRect);

    public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;

      procedure DrawButton(Canvas: TCanvas; X,Y: Integer);
      function CreateButton(AOwner: TComponent): TSpeedButton;

      // Used by Editor
      procedure BeginDragImage(Pos: TPoint);
      procedure MoveDragImage(Pos: TPoint);
      procedure EndDragImage;
      function CreateName: String;

      property CategoryIndex: Integer read FCategoryIndex write SetCategoryIdx stored False;
      property ToolbarEditor: TToolbarEditor read FToolbarEditor stored False;
      property Button: TComponent read FToolButton write SetToolButton stored StoreButton;

    published
      property Glyph: TBitmap read FGlyph write SetGlyph;
      property NumGlyphs: Integer read FNumGlyphs write FNumGlyphs default 1;
      property Enabled: Boolean read FEnabled write FEnabled default True;
      property Visible: Boolean read FVisible write FVisible default True;
      property Hint: String read FHint write FHint;
      property Description: String read FDescription write FDescription;
      property Top: Integer read FTop write FTop default 0;
      property Left: Integer read FLeft write FLeft default 0;
      property Width: Integer read FWidth write FWidth default 25;
      property Height: Integer read FHeight write FHeight default 25;
      property MenuItem: TMenuItem read FMenuItem write SetMenuItem;
      property Category: String read GetCategory write SetCategory;
      // property MultipleInstances: Boolean read FMultiInstance write FMultiInstance default False;
  end;

  TToolbarOption = (
    // toSyncWithMenu,	// Buttons wil follow menu item state
			// I have not found a way to do this yet - sorry
    toRuntimeEdit,	// Allow run-time design/edit
    toRuntimeReset,	// Display run-time Reset button
    toRuntimeHelp,	// Display run-time Help button
    toSnapToGrid,	// Snap button position to grid AFTER move
    toSnapToToolbar,	// Snap button position to toolbar DURING move
    toDragSnap,		// Snap button position to grid DURING move
    toDragImage,	// Display image of button while dragging
    toDragOutline,	// Display outline of button while dragging
    toAcceptCursor,	// Cursor state reflects drag/drop status
    toNameButtons,	// Use NameFormat when naming buttons
    toAutoAdd,		// Manage addition of buttons to toolbar
    toAutoRemove);	// Manage removal of buttons to toolbar
  TToolbarOptions = set of TToolbarOption;

  TAcceptEvent = procedure(Sender: TObject; var Accept: Boolean) of object;
  TAcceptItemEvent = procedure(Sender: TObject; Item: TToolbarItem; var Accept: Boolean) of object;
  TCreateEditorEvent = procedure(Sender: TObject; var Form: TForm) of object;

  TToolbarEditor = class(TComponent)
    private
      // Property storage
      FMenu			: TMenu;
      FToolbar			: TCustomPanel;
      FItems			: TObjectList;
      FCategories		: TStrings;
      FGridX			: Integer;
      FGridY			: Integer;
      FGridXOfs			: Integer;
      FGridYOfs			: Integer;
      FOptions			: TToolbarOptions;
      FDragCursor		: TCursor;
      FNameFormat		: String;

      // Event pointers
      FOnBeforeEdit		: TAcceptEvent;
      FOnAddButton		: TAcceptItemEvent;
      FOnMoveButton		: TAcceptItemEvent;
      FOnRemoveButton		: TAcceptItemEvent;
      FOnAfterEdit		: TNotifyEvent;
      FOnEditReset		: TNotifyEvent;
      FOnEditHelp		: TNotifyEvent;
      FOnCreateEditor		: TCreateEditorEvent;

      FDesignTimeDesigner	: TForm;

      procedure ReadData(Reader: TReader);
      procedure WriteData(Writer: TWriter);
    protected
      procedure Notification(AComponent: TComponent; Operation: TOperation); override;
      procedure DefineProperties(Filer: TFiler); override;
      procedure SetCategories(value: TStrings);
      procedure SetGrid(Index: Integer; Value: Integer);
      procedure SetGridOffset(Index: Integer; Value: Integer);
      procedure SetNameFormat(value: String);
      procedure SetEditing(value: Boolean);
      function GetEditing: Boolean;
      procedure SetOptions(Value: TToolbarOptions);
      function GetItem(Index: LongInt): TToolbarItem;
      function GetItemCount: LongInt;

      procedure CreateEditor(var FormRunToolEdit: TForm); dynamic;
      destructor Destroy; override;
    public
      constructor Create(AOwner: TComponent); override;

      procedure Edit;
      procedure SynchronizeWithMenu;
      function Add(Item: TToolbarItem): Integer;
      procedure Clear;
      procedure Delete(Index: Integer);
      function IndexOf(Item: TToolbarItem): Integer;

      // Properties
      property DesignTimeDesigner: TForm read FDesignTimeDesigner
        write FDesignTimeDesigner stored False;
      property Items[Index: LongInt]: TToolbarItem read GetItem ; Default;
      property ItemCount: LongInt read GetItemCount stored False;
      property Editing: Boolean read GetEditing Write SetEditing Stored False;

    published
      // Properties
      property Menu: TMenu read FMenu write FMenu;
      property Toolbar: TCustomPanel read FToolbar write FToolbar;
      property Categories: TStrings read FCategories write SetCategories;
      property GridX: Integer index 0 read FGridX write SetGrid default 2;
      property GridY: Integer index 1 read FGridY write SetGrid default 2;
      property GridXOffset: Integer index 0 read FGridXOfs write SetGridOffset default 0;
      property GridYOffset: Integer index 1 read FGridYOfs write SetGridOffset default 0;
      property Options: TToolbarOptions read FOptions write SetOptions;
      property DragCursor: TCursor read FDragCursor write FDragCursor Default crDrag;
      property NameFormat: String read FNameFormat write SetNameFormat;

      // Events
      property OnBeforeEdit: TAcceptEvent read FOnBeforeEdit write FOnBeforeEdit;
      property OnAddButton: TAcceptItemEvent read FOnAddButton write FOnAddButton;
      property OnMoveButton: TAcceptItemEvent read FOnMoveButton write FOnMoveButton;
      property OnRemoveButton: TAcceptItemEvent read FOnRemoveButton write FOnRemoveButton;
      property OnAfterEdit: TNotifyEvent read FOnAfterEdit write FOnAfterEdit;
      property OnEditReset: TNotifyEvent read FOnEditReset write FOnEditReset;
      property OnEditHelp: TNotifyEvent read FOnEditHelp write FOnEditHelp;
      property OnCreateEditor: TCreateEditorEvent read FOnCreateEditor write FOnCreateEditor;
  end;

  // Draggable SpeedButton
  TToolbarButton = class(TSpeedButton)
    private
      ButtonRemoved: Boolean;
      FToolbarItem: TToolbarItem;
      OriginalDragOver: TDragOverEvent;
      procedure StandInDragOver(Sender, Source: TObject; X, Y: Integer;
        State: TDragState; var Accept: Boolean);

    protected
      procedure DragOver(Source: TObject; X, Y: Integer; State: TDragState;
        var Accept: Boolean); override;
      procedure DragDrop(Source: TObject; X, Y: Integer); override;
      procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
        X, Y: Integer); override;
      procedure DoEndDrag(Target: TObject; X, Y: Integer); override;
      procedure DoStartDrag(var DragObject: TDragObject); override;
      procedure DragCanceled; override;
    public
      property ToolbarItem: TToolbarItem read FToolbarItem;

      property Align; { Why not ? }
  end;

  // DragControlObject for dragging buttons
  TDragToolObject = class(TDragControlObject)
    public
      function GetDragCursor(Accepted: Boolean; X, Y: Integer): TCursor; override;
    protected
      ToolbarItem: TToolbarItem;
    public
      constructor Create(AControl: TControl; TbItem: TToolbarItem);
  end;

  // Specialized TList clone that owns the objects it contains
  TObjectList = class(TObject)
    private
      FList: PPointerList;
      FCount: Integer;
      FCapacity: Integer;
      FOwnsObjects: Boolean;
    protected
      procedure Error; virtual;
      function Get(Index: Integer): Pointer;
      procedure Grow; virtual;
      procedure Put(Index: Integer; Item: Pointer);
      procedure SetCapacity(NewCapacity: Integer);
      procedure SetCount(NewCount: Integer);
    public
      constructor Create(Owns: Boolean);
      destructor Destroy; override;
      function Add(Item: Pointer): Integer;
      procedure Clear; virtual;
      procedure Delete(Index: Integer); virtual;
      procedure Exchange(Index1, Index2: Integer);
      function Expand: TObjectList;
      function First: Pointer;
      function IndexOf(Item: Pointer): Integer;
      procedure Insert(Index: Integer; Item: Pointer);
      function Last: Pointer;
      procedure Move(CurIndex, NewIndex: Integer);
      function Remove(Item: Pointer): Integer;
      procedure Pack;
      procedure Sort(Compare: TListSortCompare);
      property Capacity: Integer read FCapacity write SetCapacity;
      property Count: Integer read FCount write SetCount;
      property Items[Index: Integer]: Pointer read Get write Put; default;
      property List: PPointerList read FList;
      property OwnsObjects: Boolean read FOwnsObjects write FOwnsObjects default True;
  end;

  function stripShortCut(s: string): string;

const
  ToolbarEditorVersion = '01.00';

// -----------------------------------------------------------------------------
//
//			Implementation
//
// -----------------------------------------------------------------------------
implementation

uses
  dialogs,
  Consts,
  dsgnintf,
  tberuned;

// -----------------------------------------------------------------------------
// 			Support functions
// -----------------------------------------------------------------------------
function stripShortCut(s: string): string;
var
  i			: integer;
begin
  i := 1;
  while (i <= length(s)) do
    if (s[i] = '&') then
    begin
      if (i < length(s)) then
        if (s[i+1] = '&') then
          inc(i);
      delete(s,i,1)
    end else
      inc(i);
  result := s;
end;

// -----------------------------------------------------------------------------
function StripSpace(s: String): String;
var
  p			: integer;
begin
  p := pos(' ', s);
  while (p  > 0) do
  begin
    delete(s, p, 1);
    p := pos(' ', s);
  end;
  Result := s;
end;

// -----------------------------------------------------------------------------
function StripToIdent(s: String): String;
const
  Alpha = ['A'..'Z', 'a'..'z', '_'];
  AlphaNumeric = Alpha + ['0'..'9'];
var
  i			: Integer;
begin
  Result := '';
  s := StripSpace(s);
  if (Length(s) = 0) then
    exit;

  while not(s[1] in Alpha) do
  begin
    delete(s,1,1);
    if (length(s) = 0) then
      exit;
  end;

  i := 2;
  while (i <= length(s)) do
  begin
    if not(s[i] in AlphaNumeric) then
      delete(s,i,1)
    else
      inc(i);
  end;
  Result := s;
end;

// -----------------------------------------------------------------------------
//
//			TObjectList
//
// -----------------------------------------------------------------------------
procedure ListError(Ident: Integer);
begin
  raise EListError.CreateRes(Ident);
end;

procedure ListIndexError;
begin
  ListError(SListIndexError);
end;

constructor TObjectList.Create(Owns: Boolean);
begin
  Inherited Create;
  FOwnsObjects := Owns;
end;

destructor TObjectList.Destroy;
begin
  Clear;
end;

function TObjectList.Add(Item: Pointer): Integer;
begin
  Result := FCount;
  if Result = FCapacity then Grow;
  FList^[Result] := Item;
  Inc(FCount);
end;

procedure TObjectList.Clear;
var
  Index			: integer;
begin
  if (FOwnsObjects) then
    for Index := 0 to FCount-1 do
      if (Items[Index] <> nil) then
        TObject(Items[Index]).free;
  SetCount(0);
  SetCapacity(0);
end;

procedure TObjectList.Delete(Index: Integer);
begin
  if (Index < 0) or (Index >= FCount) then Error;
  if (FOwnsObjects and (Items[Index] <> nil)) then
    TObject(Items[Index]).free;
  Dec(FCount);
  if Index < FCount then
    System.Move(FList^[Index + 1], FList^[Index],
      (FCount - Index) * SizeOf(Pointer));
end;

procedure TObjectList.Error;
begin
  ListIndexError;
end;

procedure TObjectList.Exchange(Index1, Index2: Integer);
var
  Item: Pointer;
begin
  if (Index1 < 0) or (Index1 >= FCount) or
    (Index2 < 0) or (Index2 >= FCount) then Error;
  Item := FList^[Index1];
  FList^[Index1] := FList^[Index2];
  FList^[Index2] := Item;
end;

function TObjectList.Expand: TObjectList;
begin
  if FCount = FCapacity then Grow;
  Result := Self;
end;

function TObjectList.First: Pointer;
begin
  Result := Get(0);
end;

function TObjectList.Get(Index: Integer): Pointer;
begin
  if (Index < 0) or (Index >= FCount) then Error;
  Result := FList^[Index];
end;

procedure TObjectList.Grow;
var
  Delta: Integer;
begin
  if FCapacity > 8 then Delta := 16 else
    if FCapacity > 4 then Delta := 8 else
      Delta := 4;
  SetCapacity(FCapacity + Delta);
end;

function TObjectList.IndexOf(Item: Pointer): Integer;
begin
  Result := 0;
  while (Result < FCount) and (FList^[Result] <> Item) do Inc(Result);
  if Result = FCount then Result := -1;
end;

procedure TObjectList.Insert(Index: Integer; Item: Pointer);
begin
  if (Index < 0) or (Index > FCount) then Error;
  if FCount = FCapacity then Grow;
  if Index < FCount then
    System.Move(FList^[Index], FList^[Index + 1],
      (FCount - Index) * SizeOf(Pointer));
  FList^[Index] := Item;
  Inc(FCount);
end;

function TObjectList.Last: Pointer;
begin
  Result := Get(FCount - 1);
end;

procedure TObjectList.Move(CurIndex, NewIndex: Integer);
var
  Item: Pointer;
begin
  if CurIndex <> NewIndex then
  begin
    if (NewIndex < 0) or (NewIndex >= FCount) then Error;
    Item := Get(CurIndex);
    Delete(CurIndex);
    Insert(NewIndex, Item);
  end;
end;

procedure TObjectList.Put(Index: Integer; Item: Pointer);
begin
  if (Index < 0) or (Index >= FCount) then Error;
  FList^[Index] := Item;
end;

function TObjectList.Remove(Item: Pointer): Integer;
begin
  Result := IndexOf(Item);
  if Result <> -1 then Delete(Result);
end;

procedure TObjectList.Pack;
var
  I: Integer;
begin
  for I := FCount - 1 downto 0 do if Items[I] = nil then Delete(I);
end;

procedure TObjectList.SetCapacity(NewCapacity: Integer);
begin
  if (NewCapacity < FCount) or (NewCapacity > MaxListSize) then Error;
  if NewCapacity <> FCapacity then
  begin
    ReallocMem(FList, NewCapacity * SizeOf(Pointer));
    FCapacity := NewCapacity;
  end;
end;

procedure TObjectList.SetCount(NewCount: Integer);
begin
  if (NewCount < 0) or (NewCount > MaxListSize) then Error;
  if NewCount > FCapacity then SetCapacity(NewCount);
  if NewCount > FCount then
    FillChar(FList^[FCount], (NewCount - FCount) * SizeOf(Pointer), 0);
  FCount := NewCount;
end;

var
  SCompare: TListSortCompare;

procedure QuickSort(SortList: PPointerList; L, R: Integer);
var
  I, J: Integer;
  P, T: Pointer;
begin
  repeat
    I := L;
    J := R;
    P := SortList^[(L + R) shr 1];
    repeat
      while SCompare(SortList^[I], P) < 0 do Inc(I);
      while SCompare(SortList^[J], P) > 0 do Dec(J);
      if I <= J then
      begin
        T := SortList^[I];
        SortList^[I] := SortList^[J];
        SortList^[J] := T;
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then QuickSort(SortList, L, J);
    L := I;
  until I >= R;
end;

procedure TObjectList.Sort(Compare: TListSortCompare);
var
  OldCompare: TListSortCompare;
begin
  if (FList <> nil) and (Count > 0) then
  begin
    OldCompare := SCompare;
    try
      SCompare := Compare;
      QuickSort(FList, 0, Count - 1);
    finally
      SCompare := OldCompare;
    end;
  end;
end;

// -----------------------------------------------------------------------------
//
//		Drag and Drop support
//
// -----------------------------------------------------------------------------
var
    DragScreenPoint: TPoint;
    DragBackupImage: TBitmap;
    DragButtonImage: TBitmap;
const
    isDragging: Boolean = False;
    HasDrawnImage: Boolean = False;

procedure TToolbarItem.FrameRect(Handle: HDC; R: TRect);
begin
  DrawFocusRect(Handle, R);
  InflateRect(R, -1, -1);
  DrawFocusRect(Handle, R);
  InflateRect(R, -1, -1);
  DrawFocusRect(Handle, R);
  InflateRect(R, -1, -1);
  DrawFocusRect(Handle, R);
end;

procedure TToolbarItem.BeginDragImage(Pos: TPoint);
begin
  isDragging := True;

  if (toDragImage in FToolbarEditor.Options) then
  begin
    // Create images
    DragBackupImage := TBitmap.create;
    DragButtonImage := TBitmap.create;
    DragButtonImage.Width := Width;
    DragButtonImage.Height := Height;
    DragBackupImage.Width := Width;
    DragBackupImage.Height := Height;

    // Make image of button for dragging
    DrawButton(DragButtonImage.Canvas, 0, 0);
  end;

  // Draw initial image
  DrawDragImage(Pos);
end;

// -----------------------------------------------------------------------------
procedure TToolbarItem.DrawDragImage(Pos: TPoint);
var
  ScreenDC		: HDC;
  Frame			: TRect;
begin
  if ([toDragOutline, toDragImage] * FToolbarEditor.Options = []) then
    exit;
  ScreenDC := GetDC(0);
  try
    DragScreenPoint := Pos;

    // Center image around cursor
    DragScreenPoint.X := DragScreenPoint.X - (Width DIV 2);
    DragScreenPoint.Y := DragScreenPoint.Y - (Height DIV 2);

    if (toDragOutline in FToolbarEditor.Options) then
    begin
      Frame.TopLeft := DragScreenPoint;
      Frame.BottomRight := Point(DragScreenPoint.X + Width, DragScreenPoint.Y + Height);
      FrameRect(ScreenDC, Frame);
    end else
    if (toDragImage in FToolbarEditor.Options) then
    begin
      // Get backup image
      BitBlt(DragBackupImage.Canvas.Handle, 0, 0, DragBackupImage.Width, DragBackupImage.Height,
        ScreenDC, DragScreenPoint.X, DragScreenPoint.Y, SRCCOPY);
      // Draw image
      BitBlt(ScreenDC, DragScreenPoint.X, DragScreenPoint.Y, DragButtonImage.Width, DragButtonImage.Height,
        DragButtonImage.Canvas.Handle, 0, 0, SRCCOPY);
    end;
    HasDrawnImage := True;
  finally
    ReleaseDC(0, ScreenDC);
  end;
end;

// -----------------------------------------------------------------------------
procedure TToolbarItem.MoveDragImage(Pos: TPoint);
begin
  RemoveDragImage;
  DrawDragImage(Pos);
end;

// -----------------------------------------------------------------------------
procedure TToolbarItem.RemoveDragImage;
var
  ScreenDC		: HDC;
  Frame			: TRect;
begin
  if not(HasDrawnImage) or
    ([toDragOutline, toDragImage] * FToolbarEditor.Options = []) then
    exit;

  // Remove old button image
  ScreenDC := GetDC(0);
  try
    if (toDragOutline in FToolbarEditor.Options) then
    begin
      Frame.TopLeft := DragScreenPoint;
      Frame.BottomRight := Point(DragScreenPoint.X + Width, DragScreenPoint.Y + Height);
      FrameRect(ScreenDC, Frame);
    end else
    if (toDragImage in FToolbarEditor.Options) then
    begin
      // Restore backup image
      BitBlt(ScreenDC, DragScreenPoint.X, DragScreenPoint.Y, DragBackupImage.Width, DragBackupImage.Height,
        DragBackupImage.Canvas.Handle, 0, 0, SRCCOPY);
    end;
    HasDrawnImage := False;
  finally
    ReleaseDC(0, ScreenDC);
  end;
end;

// -----------------------------------------------------------------------------
procedure TToolbarItem.EndDragImage;
begin
  if ([toDragOutline, toDragImage] * FToolbarEditor.Options = []) then
    exit;
  if (isDragging) then
  begin
    RemoveDragImage;
    isDragging := False;

    if (toDragImage in FToolbarEditor.Options) then
    begin
      DragBackupImage.Free;
      DragButtonImage.Free;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Handle drag of buttons on speedbar
//
procedure TToolbarButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  if (ToolbarItem.ToolbarEditor.Editing) then
  begin
    ToolbarItem.FMoveState := msMoving;
    BeginDrag(True)
  end else
    inherited MouseDown(Button, Shift, X, Y);
end;

procedure TToolbarButton.DoStartDrag(var DragObject: TDragObject);
var
  Pos			: TPoint;
begin
  // Hide button while it is being dragged
  if (toDragImage in ToolbarItem.ToolbarEditor.Options) then
  begin
    hide;
    update;
  end;

  // Replace toolbar drag over event handler
  OriginalDragOver := TPanel(ToolbarItem.ToolbarEditor.Toolbar).OnDragOver;
  TPanel(ToolbarItem.ToolbarEditor.Toolbar).OnDragOver := StandInDragOver;

  DragObject := TDragToolObject.Create(Self, ToolbarItem);
  GetCursorPos(Pos);
  ToolbarItem.BeginDragImage(Pos);
end;

procedure TToolbarButton.DoEndDrag(Target: TObject; X, Y: Integer);
var
  Pos			,
  SavePos		: TPoint;
  Accept		: Boolean;
  DeleteButton		: Boolean;
begin
  // Restore toolbar drag over event handler
  TPanel(ToolbarItem.ToolbarEditor.Toolbar).OnDragOver := OriginalDragOver;

  ToolbarItem.EndDragImage;
  DeleteButton := False;
  // Was button removed? (signalled in DragCanceled)
  if (ButtonRemoved) then
  begin
    ButtonRemoved := False;
    if (ToolbarItem.FMoveState = msMoving) then
    begin // Button was moved off toolbar
      Accept := (toAutoRemove in ToolbarItem.ToolbarEditor.Options);
      // Send OnRemoveButton event
      if (Assigned(ToolbarItem.ToolbarEditor.OnRemoveButton)) then
        ToolbarItem.ToolbarEditor.OnRemoveButton(ToolbarItem.ToolbarEditor,
          ToolbarItem, Accept);
    end else // Button was moved from editor but not accepted by toolbar
      Accept := True;

    // Delete button if remove was accepted
    DeleteButton := Accept;
  end else
  begin
    Pos := (Parent as TControl).ScreenToClient(Point(ToolbarItem.Left,ToolbarItem.Top));
    SavePos.X := Left;
    SavePos.Y := Top;
    Left := Pos.X;
    Top := Pos.Y;
    if (ToolbarItem.FMoveState = msMoving) then
    begin // Button was moved on the toolbar
      // Send OnMoveButton event
      Accept := True;
      if (Assigned(ToolbarItem.ToolbarEditor.OnMoveButton)) then
        ToolbarItem.ToolbarEditor.OnMoveButton(ToolbarItem.ToolbarEditor,
          ToolbarItem, Accept);
      // Restore original position if move was rejected
      if not(Accept) then
      begin
        Left := SavePos.X;
        Top := SavePos.Y;
      end;
    end else
    begin // Button was moved from editor and accepted by toolbar
      Accept := (toAutoAdd in ToolbarItem.ToolbarEditor.Options);
      // Send OnAddButton event
      if (Assigned(ToolbarItem.ToolbarEditor.OnAddButton)) then
        ToolbarItem.ToolbarEditor.OnAddButton(ToolbarItem.ToolbarEditor,
          ToolbarItem, Accept);
      // Delete button if add was rejected
      DeleteButton := not(Accept)
    end;
  end;

  // Delete or Display button again
  if (DeleteButton) then
  begin
    ToolbarItem.Button := nil;
    ToolbarItem.FMoveState := msAdding;
    Free;
  end else
  if (toDragImage in ToolbarItem.ToolbarEditor.Options) then
  begin
    show;
    update;
  end;
end;

// -----------------------------------------------------------------------------
// Handle drag of buttons away from speedbar = Remove
//
procedure TToolbarButton.DragCanceled;
begin
  ButtonRemoved := True;
end;

// -----------------------------------------------------------------------------
// Handle drop on top of buttons on speedbar
// Ignore and let parent handle event.
procedure TToolbarButton.DragOver(Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  inherited DragOver(Source, X, Y, State, Accept);
  if (Accept) then
    exit;

  // Let buttons parent handle event
  if (Parent is TControl) then
    if (Assigned(TToolbarButton(Parent).OnDragOver)) then
      TToolbarButton(Parent).OnDragOver(Parent, Source, X+Left, Y+Top, State, Accept);
end;

procedure TToolbarButton.DragDrop(Source: TObject; X, Y: Integer);
begin
  if (Assigned(OnDragDrop)) then
    inherited DragDrop(Source, X, Y)
  else
    // Let buttons parent handle event
    if (Parent is TControl) then
      if (Assigned(TToolbarButton(Parent).OnDragDrop)) then
        TToolbarButton(Parent).DragDrop(Source, X+Left, Y+Top);
end;

// -----------------------------------------------------------------------------
procedure TToolbarButton.StandInDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  // If we are allowed to handle the drop automatically
  // then do so...
  Accept := (toAutoAdd in ToolbarItem.ToolbarEditor.Options) and (Source = self);

  // Give original handler a chance to reject drop
  if (Assigned(OriginalDragOver)) then
    OriginalDragOver(Sender, Source, X, Y, State, Accept);
end;

// -----------------------------------------------------------------------------
//
//			TDragToolObject
//
// -----------------------------------------------------------------------------
function TDragToolObject.GetDragCursor(Accepted: Boolean; X, Y: Integer): TCursor;
var
  SnapX			,
  SnapY			: Integer;
  SnapPoint		: TPoint;
  DrawPoint		: TPoint;
  TBRect		: TRect;
begin
  SnapX := X;
  SnapY := Y;
  // Snap to Toolbar
  if (toSnapToToolbar in ToolbarItem.ToolbarEditor.Options)
    and (Accepted) and (ToolbarItem.FMoveState = msMoving) then
  begin
    TBRect := ToolbarItem.ToolbarEditor.Toolbar.ClientRect;
    TBRect.TopLeft := ToolbarItem.ToolbarEditor.Toolbar.ClientToScreen(TBRect.TopLeft);
    TBRect.BottomRight := ToolbarItem.ToolbarEditor.Toolbar.ClientToScreen(TBRect.BottomRight);
    InflateRect(TBRect, -ToolbarItem.Width DIV 2, -ToolbarItem.Height DIV 2);
    if (SnapX < TBRect.Left) then
      SnapX := TBRect.Left
    else if (SnapX > TBRect.Right) then
      SnapX := TBRect.Right;
    if (SnapY < TBRect.Top) then
      SnapY := TBRect.Top
    else if (SnapY > TBRect.Bottom) then
      SnapY := TBRect.Bottom;
  end;

  // Snap to grid
  if ([toDragSnap, toSnapToGrid] * ToolbarItem.ToolbarEditor.Options <> []) then
  begin
    SnapPoint.Y := (SnapY + (ToolbarItem.ToolbarEditor.GridY DIV 2) - ToolbarItem.ToolbarEditor.GridYOffset)
      DIV ToolbarItem.ToolbarEditor.GridY * ToolbarItem.ToolbarEditor.GridY +
      ToolbarItem.ToolbarEditor.GridYOffset;
    SnapPoint.X := (SnapX + (ToolbarItem.ToolbarEditor.GridX DIV 2) - ToolbarItem.ToolbarEditor.GridXOffset)
      DIV ToolbarItem.ToolbarEditor.GridX * ToolbarItem.ToolbarEditor.GridX +
      ToolbarItem.ToolbarEditor.GridXOffset;
  end else
  begin
    SnapPoint.X := SnapX;
    SnapPoint.Y := SnapY;
  end;

  if (toDragSnap in ToolbarItem.ToolbarEditor.Options) then
    DrawPoint := SnapPoint
  else
  begin
    DrawPoint.X := SnapX;
    DrawPoint.Y := SnapY;
  end;

  // Redraw image
  if (ToolbarItem.FMoveState = msMoving) and not(Accepted) then
    // Hide image if moving an existing button off the toolbar
    ToolbarItem.RemoveDragImage
  else
    ToolbarItem.MoveDragImage(DrawPoint);

  // Center Button around cursor
  ToolbarItem.Left := SnapPoint.X - (ToolbarItem.Width DIV 2);
  ToolbarItem.Top := SnapPoint.Y - (ToolbarItem.Height DIV 2);

  if not(toAcceptCursor in ToolbarItem.ToolbarEditor.Options) then
    Accepted := True;
  result := inherited GetDragCursor(Accepted, X, Y);
end;

// -----------------------------------------------------------------------------
constructor TDragToolObject.Create(AControl: TControl; TbItem: TToolbarItem);
begin
  inherited Create(AControl);
  ToolbarItem := TbItem;
end;

// -----------------------------------------------------------------------------
//
//			TToolbarItem
//
// -----------------------------------------------------------------------------
constructor TToolbarItem.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  // This can safely be removed. FToolbarEditor is set in TToolbarEdit.Add().
  if (AOwner is TToolbarEditor) then
    FToolbarEditor := TToolbarEditor(AOwner);

  FCategoryIndex := 0;
  FGlyph := TBitmap.create;
  FNumGlyphs := 1;
  FEnabled := True;
  FVisible := True;
  FTop := 0;
  FLeft := 0;
  FWidth := 25;
  FHeight := 25;
  FToolButton := nil;
  FMoveState := msAdding;
end;

// -----------------------------------------------------------------------------
destructor TToolbarItem.Destroy;
begin
  if (FGlyph <> nil) then
    FGlyph.free;
  inherited Destroy;
end;

// -----------------------------------------------------------------------------
procedure TToolbarItem.DrawButton(Canvas: TCanvas; X,Y: Integer);
var
  PaintRect		: TRect;
  PenColor		: TColor;
  BrushColor		: TColor;
  BrushStyle		: TBrushStyle;
  OneWidth		: Integer;
begin
  // Calculate button rect
  PaintRect.Top := Y;
  PaintRect.Left := X;
  PaintRect.Bottom := Y + Height;
  PaintRect.Right := X + Width;

  // Save Canvas properties
  PenColor := Canvas.Pen.Color;
  BrushColor := Canvas.Brush.Color;
  BrushStyle := Canvas.Brush.Style;
  try

    // Draw the button rectangle
    PaintRect := DrawButtonFace(Canvas, PaintRect, 1, bsNew, False, False, False);

    // Draw the button bitmap...
    if not(Glyph.Empty) then
    begin
      // Center glyph
      OneWidth :=  Glyph.Width DIV NumGlyphs;
      PaintRect.Top := PaintRect.Top +
        ((PaintRect.Bottom - PaintRect.Top) - Glyph.Height) DIV 2;
      PaintRect.Bottom := PaintRect.Top + Glyph.Height;
      PaintRect.Left := PaintRect.Left +
        ((PaintRect.Right - PaintRect.Left) - OneWidth) DIV 2;
      PaintRect.Right := PaintRect.Left + OneWidth;
      // Draw transparent glyph
      Canvas.BrushCopy(PaintRect, Glyph, Rect(0,0,Glyph.Height, OneWidth),
        Glyph.TransparentColor);
    end;

  finally
    // Restore Canvas properties
    Canvas.Pen.Color := PenColor;
    Canvas.Brush.Color := BrushColor;
    Canvas.Brush.Style := BrushStyle;
  end;
end;

// -----------------------------------------------------------------------------
// TToolbarItem Property access
procedure TToolbarItem.SetGlyph(value: TBitmap);
begin
  FNumGlyphs := Value.Width div Value.Height;
  if (FNumGlyphs > 4) then
    FNumGlyphs := 1;
  if (FGlyph = nil) then
    FGlyph := TBitmap.create;
  FGlyph.assign(value);
end;

// -----------------------------------------------------------------------------
procedure TToolbarItem.SetToolButton(value: TComponent);
begin
  if (FToolButton = value) then
    exit;
  FToolButton := value;
end;

// -----------------------------------------------------------------------------
function TToolbarItem.StoreButton: boolean;
begin
  Result := False;
end;

// -----------------------------------------------------------------------------
procedure TToolbarItem.SetCategoryIdx(idx: Integer);
begin
  if (idx < ToolbarEditor.Categories.Count) then
    FCategoryIndex := idx;
end;

// -----------------------------------------------------------------------------
procedure TToolbarItem.SetCategory(value: String);
var
  newCatIdx		: integer;
begin
  newCatIdx := ToolbarEditor.Categories.IndexOf(value);
  if (newCatIdx = -1) then
    newCatIdx := ToolbarEditor.Categories.Add(value);
  CategoryIndex := newCatIdx;
end;

function TToolbarItem.GetCategory: String;
begin
  Result := ToolbarEditor.Categories[FCategoryIndex];
end;

// -----------------------------------------------------------------------------
function TToolbarItem.CreateName: String;
var
  index			: integer;
  s			: String;
  StripFormat		,
  StripCategory		,
  StripDescription	: String;
begin
  StripFormat := StripSpace(ToolbarEditor.NameFormat);
  StripCategory := StripToIdent(Category);
  StripDescription := StripToIdent(Description);
  try
    index := 1;
    // Generate name
    try
      s := format(StripFormat, [StripCategory, StripDescription, Index]);
      if not(IsValidIdent(s)) then
        raise Exception.Create('Cannot make name from: '+StripFormat);
    except
      raise Exception.Create('Bad NameFormat: '+#13+
        ToolbarEditor.NameFormat+
        ' ['+StripCategory+','+StripDescription+','+IntToStr(Index)+']');
    end;
    // Try generated name
    try
      Name := s;
    except
      // Failed - Append numeric postfix
      if (s = format(StripFormat, [StripCategory, StripDescription, 0])) then
        StripFormat := StripFormat+'%2:d';
      while (True) do
      begin
        // Generate and try new name
        try
          Name := format(StripFormat, [StripCategory, StripDescription, Index]);
          // Got it!
          break;
        except
          // Give user a chance to abort
          if (index MOD 100 = 0) then
            Application.ProcessMessages;
          // Try next numeric postfix
          inc(Index);
          // Check for overflow
          if (index > 1000) then
            raise Exception.Create('Failed to set name using: '+ToolbarEditor.NameFormat);
        end;
      end;
    end;
  finally
    Result := Name;
  end;
end;

// -----------------------------------------------------------------------------
function TToolbarItem.CreateButton(AOwner: TComponent): TSpeedButton;
var
  NewButton		: TSpeedButton;
  Pos			: TPoint;
begin
  if (Button <> nil) then
    NewButton := Button as TSpeedButton
  else
  begin
    NewButton := TToolbarButton.Create(AOwner);
    NewButton.Parent := AOwner as TWinControl;
    Button := NewButton;
  end;

  if (NewButton is TToolbarButton) then
    TToolbarButton(NewButton).FToolbarItem := self;
  TToolbarButton(NewButton).DragCursor := ToolbarEditor.DragCursor;
  NewButton.Height := Height;
  NewButton.Width := Width;
  Pos := (AOwner as TControl).ScreenToClient(Point(Left,Top));
  NewButton.Top := Pos.Y;
  NewButton.Left := Pos.X;

  NewButton.Hint := Hint;
  NewButton.Glyph := Glyph;
  NewButton.NumGlyphs := NumGlyphs;
  NewButton.Enabled := Enabled or ToolbarEditor.Editing;
  NewButton.Visible := Visible or ToolbarEditor.Editing;
  if (MenuItem <> nil) then
    NewButton.OnClick := MenuItem.OnClick;
  Result := TSpeedButton(NewButton);
end;

procedure TToolbarItem.SetMenuItem(Value: TMenuItem);
begin
  FMenuItem := Value;
end;

// -----------------------------------------------------------------------------
//
//			TToolbarEditor
//
// -----------------------------------------------------------------------------
constructor TToolbarEditor.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FItems := TObjectList.Create(True);
  FCategories := TStringList.create;
  FGridX := 2;
  FGridY := 2;
  FGridXOfs := 0;
  FGridYOfs := 0;
  FDragCursor := crDrag;
  FOptions := [toRuntimeEdit, toDragSnap, toSnapToToolbar, toDragImage, toAcceptCursor];
  FNameFormat := 'tb%0:s%1:s%2:d';
end;

// -----------------------------------------------------------------------------
destructor TToolbarEditor.Destroy;
begin
  FCategories.free;
  FItems.free;
  inherited Destroy;
end;

// -----------------------------------------------------------------------------
// TToolbarEditor Persistence
// -----------------------------------------------------------------------------
procedure TToolbarEditor.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineProperty('Buttons', ReadData, WriteData, (ItemCount > 0));
end;

// -----------------------------------------------------------------------------
// Read items from stream
procedure TToolbarEditor.ReadData(Reader: TReader);
var
  item			: TToolbarItem;
begin
  Clear;
  Reader.readListBegin;
  while not(Reader.endOfList) do
  begin
    item := TToolbarItem.create(self);
    Reader.ReadComponent(item);
    Add(item);
  end;
  Reader.readListEnd;
end;

// -----------------------------------------------------------------------------
// Write items to stream
procedure TToolbarEditor.WriteData(Writer: TWriter);
var
  i			: integer;
begin
  FItems.pack;
  Writer.writeListBegin;
  try
    for i:= 0 to ItemCount-1 do
      if (Items[i] <> nil) then
        Writer.WriteComponent(Items[i]);
  finally
    Writer.writeListEnd;;
  end;
end;

// -----------------------------------------------------------------------------
// TToolbarEditor notification
// -----------------------------------------------------------------------------
procedure TToolbarEditor.Notification(AComponent: TComponent; Operation: TOperation);
var
  i			: integer;
begin
  inherited Notification(AComponent, Operation);

  if (Operation = opRemove) then
  begin
//    if (AComponent is TSpeedButton) then
//      showMessage('Component removed: '+AComponent.Name);
    if (AComponent = Toolbar) then
      Toolbar := nil;
    if (AComponent = Menu) then
      Menu := nil;

    // Also handle notification for ToolbarItems
    for i:=0 to ItemCount-1 do
    begin
      if (Items[i].Button <> nil) and (Items[i].Button = AComponent) then
        Items[i].Button := nil;
      if (Items[i].MenuItem <> nil) and (Items[i].MenuItem = AComponent) then
        Items[i].MenuItem := nil;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// TToolbarEditor property access
// -----------------------------------------------------------------------------
procedure TToolbarEditor.SetCategories(value: TStrings);
var
  i			: integer;
  newIdx		: integer;
begin
  if not(csLoading in ComponentState) then
    // Make sure that all used categories are in the new list and...
    // renumber all items category indexes
    for i:=0 to ItemCount-1 do
    begin
      newIdx := value.IndexOf(Items[i].Category);
      // Add category if missing
      if (newIdx = -1) then
        newIdx := value.Add(Items[i].Category);
      // Use new category index
      Items[i].CategoryIndex := newIdx;
    end;

  FCategories.Assign(value);
end;

// -----------------------------------------------------------------------------
procedure TToolbarEditor.SetGrid(Index: Integer; Value: Integer);
begin
  if (Value < 1) then
    exit;

  case (Index) of
    0: FGridX := Value;
    1: FGridY := Value;
  end;
end;

// -----------------------------------------------------------------------------
procedure TToolbarEditor.SetGridOffset(Index: Integer; Value: Integer);
begin
  if (Value < 0) then
    exit;

  case (Index) of
    0: FGridXOfs := Value;
    1: FGridYOfs := Value;
  end;
end;

// -----------------------------------------------------------------------------
procedure TToolbarEditor.SetNameFormat(value: String);
begin
  value := StripSpace(value);
  // Minimal check of validity
  if IsValidIdent(format(value, ['Category', 'Description', 1])) then
    FNameFormat := value;
end;

// -----------------------------------------------------------------------------
// TToolbarEditor run time design
// -----------------------------------------------------------------------------
procedure TToolbarEditor.Edit;
var
  AcceptEdit		: Boolean;
  i			: integer;
begin
  if (csDesigning in ComponentState) or not(toRuntimeEdit in FOptions)
    or (Toolbar = nil) then
    exit;

  // Create OnBeforeEdit event
  AcceptEdit := True;
  if (Assigned(FOnBeforeEdit)) then
    FOnBeforeEdit(Self, AcceptEdit);
  if not(AcceptEdit) then
    exit;

  // Create run time editor
  CreateEditor(TForm(FRuntimeToolbarEditor));

  // Let's rock!
  FRuntimeToolbarEditor.Show;

  // Enable all buttons for edit
  for i:= 0 to ItemCount-1 do
    if (Items[i].Button <> nil) then
    begin
      TSpeedButton(Items[i].Button).Enabled := True;
      TSpeedButton(Items[i].Button).Visible := True;
    end;
end;

// -----------------------------------------------------------------------------
procedure TToolbarEditor.CreateEditor(var FormRunToolEdit: TForm);
begin
  // Only create one editor at a time
  if (FormRunToolEdit = nil) then
  begin
    // Let user create custom editor
    if (Assigned(FOnCreateEditor)) then
    begin
      FOnCreateEditor(self, FormRunToolEdit);
      { Make sure that what we get is valid }
      if (FormRunToolEdit <> nil) then
        if not(FormRunToolEdit is TFormRunToolEdit) then
        begin
          { Can't use it - kill form }
          FormRunToolEdit.Free;
          FormRunToolEdit := nil;
          raise Exception.Create('Invalid form type returned from OnCreateEditor');
        end;
    end;
    // Create default editor if user didn't create one
    if (FormRunToolEdit = nil) then
      FormRunToolEdit := TFormDefaultToolbarEditor.Create(Self);

    TFormRunToolEdit(FormRunToolEdit).ToolbarEditor := self;
  end;
end;

// -----------------------------------------------------------------------------
procedure TToolbarEditor.SetEditing(value: Boolean);
begin
  if (Value = GetEditing) then
    exit;
  if (Value) then
    edit
  else
    FRuntimeToolbarEditor.Close;
end;

function TToolbarEditor.GetEditing: Boolean;
begin
  Result := (FRuntimeToolbarEditor <> nil);
end;

// -----------------------------------------------------------------------------
procedure TToolbarEditor.SetOptions(Value: TToolbarOptions);
begin
  if (Value = FOptions) then
    exit;
  // toDragImage and toDragOutline are mutually exclusive.
  // Find out if any of them have been added and remove
  // the other if so.
  if (toDragImage in Value) and not(toDragImage in FOptions) then
    Value := Value - [toDragOutline];
  if (toDragOutline in Value) and not(toDragOutline in FOptions) then
    Value := Value - [toDragImage];

  FOptions := Value;
end;

// -----------------------------------------------------------------------------
function TToolbarEditor.GetItem(Index: LongInt): TToolbarItem;
begin
  result := TToolbarItem(FItems[Index]);
end;

// -----------------------------------------------------------------------------
function TToolbarEditor.GetItemCount: LongInt;
begin
  Result := FItems.Count;
end;

// -----------------------------------------------------------------------------
function TToolbarEditor.Add(Item: TToolbarItem): Integer;
begin
  Item.FToolbarEditor := self;
  Result := FItems.Add(Item);
end;

// -----------------------------------------------------------------------------
procedure TToolbarEditor.Clear;
begin
  FItems.Clear;
end;

// -----------------------------------------------------------------------------
procedure TToolbarEditor.Delete(Index: Integer);
begin
  FItems.Delete(Index);
end;

// -----------------------------------------------------------------------------
function TToolbarEditor.IndexOf(Item: TToolbarItem): Integer;
begin
  Result := FItems.IndexOf(Item);
end;

// -----------------------------------------------------------------------------
// Manual Menu Syncronization
procedure TToolbarEditor.SynchronizeWithMenu;
var
  i			: integer;
begin
  for i:=0 to ItemCount-1 do
  begin
    if (Items[i].MenuItem <> nil) then
    begin
      Items[i].Enabled := Items[i].MenuItem.Enabled;
      Items[i].Visible := Items[i].MenuItem.Visible;
    end;
    if (Items[i].Button <> nil) then
    begin
      TSpeedButton(Items[i].Button).Enabled := Items[i].Enabled;
      TSpeedButton(Items[i].Button).Visible := Items[i].Visible;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Automatic Menu Syncronization
(* SORRY NOT IMPLEMENTED YET - DON'T KNOW HOW TO! *)

// -----------------------------------------------------------------------------
//
//			Registration
//
// -----------------------------------------------------------------------------
procedure RegisterToolbarClass;
begin
  RegisterClass(TToolbarItem);
end;

// -----------------------------------------------------------------------------
Begin
  RegisterToolbarClass;
end.

// -----------------------------------------------------------------------------
// To do:
// * OnChange event property
// * RenameCategory method
// * Support for all types of controls on toolbar.
//   E.g. Normal TSpeedButton and TComboBox etc.
// * Drag and Drop functionality should be moved out of the TToolbarButton class.
//   This must be done before any other button types can be supported.
// * Toolbar persistence.
//   Save & Load layout to/from registry.
// * Automatic syncronization of button state with menu item state.
//   If a menu item is enable/disabled the corresponding button should
//   automatically be enabled/disabled.
// * Implement TToolbarItem.MultipleInstances.
//   If True, a new button will be created each time the item is dragged from
//   the run time editor. Can be used to implement Spacer items.
// * Implement SnapToGrid procedure. Remove Snap functionality from
//   TDragToolObject.GetDragCursor.
// * CR key in Toolbar designer should move focus to Object Inspector.
//   Probably done with ActivateInspector.
// -----------------------------------------------------------------------------
// Known bugs:
// * If form has a TToolbarEditor that contains one or more TToolbarItems,
//   the form can not be loaded into the Dephi IDE editor as text.
// * Bug in "Snap to grid" calculations.
// * The Notification mechanism does not work as expected.
// -----------------------------------------------------------------------------

